/*
 * $QNXLicenseC:
 * Copyright 2021, QNX Software Systems.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"). You
 * may not reproduce, modify or distribute this software except in
 * compliance with the License. You may obtain a copy of the License
 * at: http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTIES OF ANY KIND, either express or implied.
 *
 * This file may contain contributions from others, either as
 * contributors under the License or as licensors under other terms.
 * Please review this entire file for other proprietary rights or license
 * notices, as well as the QNX Development Suite License Guide at
 * http://licensing.qnx.com/license-guide/ for other information.
 * $
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <stdbool.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef UNITTEST
#include <hw/i2c.h>
#include <devctl.h>
#include <pthread.h>
#include <sys/select.h>
#include <sys/slog2.h>
#include <sys/mman.h>
#include <sys/neutrino.h>
#include "utils.h"
#include <input/mtouch_driver.h>
#include "input/mtouch_client.h"
#include "input/parseopts.h"
#include "input/inputtrace.h"
#include "input/mtouch_log.h"
#include "input/mtouch_params.h"
#include "input/event_types.h"
#include "input/calib.h"
#include "i2c_client.h"
#else
#include "touchscreen_header.h"
#include "interrupt_header.h"
#include "err_mem_header.h"
#endif

#include "himax_platform.h"
#include "fidm_touch.h"

#ifndef UNITTEST
#define himax_static static
#else
#define himax_static
#endif 

#define HIMAX_DRIVER_VER "Himax_QNX_SampleCode_0922"

#define ERROR                -1
#define CHECKSUM_FAIL        -2
#define MAX_INIT_RETRIES       10
#define MAX_HW_INIT_RETRIES    5
#define CONTACT_UP            0
#define CONTACT_DOWN        1
#define CONTACT_MOVE        2
#define TOUCH_INFO_POINT_CNT 52
#define MAX_RETRY_ATTEMPTS  10
#define MAX_HW_RETRY_ATTEMPTS  2 /*for cold boot retry*/



/***************************************************************************************************
*   Static Functions' prototypes
***************************************************************************************************/
himax_static int set_option(const char* option, const char* value, void* arg);
himax_static int get_contact_id(void* packet, uint8_t digit_idx, uint32_t* contact_id, void* arg);
himax_static int is_contact_down(void* packet, uint8_t digit_idx, int* valid, void* arg);
himax_static int get_coords(void* packet, uint8_t digit_idx, int32_t* x, int32_t* y, void* arg);
/***************************************************************************************************/

/* Write critical logs into errmem */
void error_memory(const char * fmt, ...)
{
#define MAXLINE    1024   /* max text line length */

    va_list ap;
    char buf[MAXLINE] = {0};

    va_start(ap, fmt);
    vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    vWritePrintfErrmem(buf);

}

int get_down_count(void *packet, uint32_t *down_count, void *arg)
{
#if 0
     himax_dev_t *himax = arg;

    if (himax) {
        md = &himax->md;
        mtouch_info("Himax", "prev_records %d\n", md->num_prv_rec);
        *down_count = md->num_prv_rec;
    }
#else
    *down_count = 1;
#endif
    return 0;
}

void himax_release_touch(himax_dev_t *himax)
{
    int idx;
    int pending_release = 0;

    if (himax == NULL)
        return;

    for (idx = 0; idx < HIMAX_MAX_NUM_TOUCHPTS; idx++)
    {
        touch_state_t *touch_state = &himax->state[idx];
        // Check if the touch point is active
        if (touch_state->is_valid && (touch_state->contact & (CONTACT_DOWN | CONTACT_MOVE)))
        {
            pending_release++;
            // Reset the touch state
            touch_state->contact = 0;
            himax->prev_touch_state[idx] = 0;
            mtouch_info(himax->log_name, "Releasing touch [%d] Coordinate X: %d Y: %d Status: %s\n",
                        idx,
                        touch_state->x,
                        touch_state->y,
                        (touch_state->contact ? "Pressed" : "Released"));
        }
    }
    // If there is any pending release, process the packet
    if (pending_release > 0) {
        mtouch_info(himax->log_name, "release_pending_touches entered - Pending Release - %d\n", pending_release);
        mtouch_driver_process_packet(himax->inputevents_hdl, himax->state, himax, MTOUCH_PARSER_FLAG_NONE);
        mtouch_info(himax->log_name, "Released all %d touch pointers by process packet\n", pending_release);
    }
}

himax_static int is_contact_down(void* packet, uint8_t digit_idx, int* valid, void* arg)
{
    himax_dev_t *himax = arg;
    touch_state_t* touch_state;
    int idx;
    *valid = 0;

    if (himax) {
        idx = digit_idx;
        touch_state = &himax->state[idx];
        if (touch_state->is_valid) {
            if ((touch_state->contact & CONTACT_DOWN) || (touch_state->contact & CONTACT_MOVE))
            {
                *valid = 1;
            }
            else {
                *valid = 0;
            }
            if (himax->prev_touch_state[idx] != touch_state->contact) {
                himax->prev_touch_state[idx] = touch_state->contact;
                mtouch_info(himax->log_name, "Touch [%d] : %s\n", digit_idx, (*valid)? "TOUCH_DOWN":"TOUCH_UP");
            }
        } else {
            mtouch_error(himax->log_name, "No touch with finger id %d\n", digit_idx);
            error_memory("Himax_Touch: No touch with finger id %d\n", digit_idx);
            return 0;
        }
    } else {
        mtouch_error("Himax", "Invalid device\n");
        error_memory("Himax_Touch: Invalid device\n");
        return ERROR;
    }

    return 0;
}

himax_static int get_contact_id(void* packet, uint8_t digit_idx, uint32_t* contact_id, void* arg)
{
    himax_dev_t *himax = arg;
    int idx;

    if (himax) {
        idx = digit_idx;
        if (himax->state[idx].is_valid) {
            //*contact_id = digit_idx;
            *contact_id = himax->state[digit_idx].t;
            if (himax->verbose > 6) {
                mtouch_info(himax->log_name, "touch with finger %d\n", *contact_id);
            }
        } else {
            mtouch_error(himax->log_name, "No touch with finger %d\n", digit_idx);
            error_memory("Himax_Touch: No touch with finger %d\n", digit_idx);
            return ERROR;
        }
    } else {
        mtouch_error("Himax", "Invalid device\n");
        error_memory("Himax_Touch: Invalid device\n");
        return ERROR;
    }

    return 0;
}

#if 0
static void get_seq_id(void* packet, uint32_t* seq_id, void* arg)
{
    himax_dev_t *himax = arg;

    if (himax) {
        *seq_id = himax->seq_id++;
        himax->seq_id &= ~(INPUTTRACE_SEQ_TYPE(INPUTTRACE_SEQ_TYPE_MASK));
    } else {
        mtouch_error("Himax", "Invalid device\n");
        return;
    }
}

static int get_contact_type(void *packet, uint8_t digit_idx, uint32_t *contact_type, void *arg)
{
    himax_dev_t *himax = arg;
    touch_state_t* touch_state;
    int idx;

    if (himax) {
        idx = digit_idx;
        touch_state = &himax->state[idx];
        if (touch_state->is_valid) {
            mtouch_info(himax->log_name, "idx %d type %d : valid State\n", idx, touch_state->type);
            *contact_type = touch_state->type;
        } else {
            return ERROR;
        }
    } else {
        mtouch_error("Himax", "Invalid device\n");
        return ERROR;
    }

    return 0;
}
#endif

himax_static int get_coords(void* packet, uint8_t digit_idx, int32_t* x, int32_t* y, void* arg)
{
    himax_dev_t *himax = arg;
    touch_state_t* touch_state;
    int idx;

    if (himax) {
        idx = digit_idx;
        touch_state = &himax->state[idx];
        if (touch_state->is_valid) {
            *x = touch_state->x;
            *y = touch_state->y;
            mtouch_info(himax->log_name, "Touch [%d] : (x , y) = (%-4d, %-4d)\n", digit_idx, *x, *y);
        } else {
            mtouch_error(himax->log_name, "No touch with finger %d\n", digit_idx);
            error_memory("Himax_Touch: No touch with finger %d\n", digit_idx);
            return ERROR;
        }
    } else {
        mtouch_error("Himax", "Invalid device\n");
        error_memory("Himax_Touch: Invalid device\n");
        return ERROR;
    }

    return EOK;
}

int get_touch_pressure(void *packet, uint8_t digit_idx, uint32_t *touch_pressure, void *arg)
{
    himax_dev_t *himax = arg;
    touch_state_t* touch_state;
    int idx;

    if (himax) {
        idx = digit_idx;
        touch_state = &himax->state[idx];
        if (touch_state->is_valid) {
            *touch_pressure = touch_state->pressure;
        } else {
            mtouch_error(himax->log_name, "No touch with finger %d\n",digit_idx);
            error_memory("Himax_Touch: No touch with finger %d\n",digit_idx);
            return ERROR;
        }
    } else {
        mtouch_error("Himax", "Invalid device\n");
        error_memory("Himax_Touch: Invalid device\n");
        return ERROR;
    }

    return EOK;
}

int get_touch_orientation(void *packet, uint8_t digit_idx, uint32_t *touch_orientation, void *arg)
{
    himax_dev_t *himax = arg;
    touch_state_t* touch_state;
    int idx;

    if (himax) {
        idx = digit_idx;

        touch_state = &himax->state[idx];

        if (touch_state->is_valid) {
            *touch_orientation = touch_state->orientation;
        } else {
            mtouch_error(himax->log_name, "No touch with finger %d\n",digit_idx);
            error_memory("Himax_Touch: No touch with finger %d\n",digit_idx);
            return ERROR;
        }
    } else {
        mtouch_error("Himax", "Invalid device\n");
        error_memory("Himax_Touch: Invalid device\n");
        return ERROR;
    }

    return EOK;
}

int himax_mtouch_register_pm(himax_dev_t *himax)
{
    struct pm_register_s pm_reg;
    int ret;

    himax->pm_fd = open(himax->pm_dev, O_RDWR  | O_CLOEXEC);
    if (himax->pm_fd  == -1) {
        mtouch_error(himax->log_name, "dev pm open() failed %d", strerror(errno));
        error_memory("Himax_Touch: dev pm open() failed %s", strerror(errno));
        goto fail;
    }

    memset(&pm_reg, 0x0, sizeof(struct pm_register_s));
    INIT_PM_REGISTER_STRUCT(&pm_reg);

    strlcpy(pm_reg.name, himax->log_name, sizeof(pm_reg.name));
    pm_reg.pulse_codes[PM_STATE_PREPARE] = HIMAX_PM_PREPARE_PULSE;
    pm_reg.pulse_codes[PM_STATE_SUSPEND] = HIMAX_PM_SUSPEND_PULSE;
    pm_reg.pulse_codes[PM_STATE_RESUME] = HIMAX_PM_RESUME_PULSE;
    pm_reg.pulse_codes[PM_STATE_COMPLETE] = -1;
    pm_reg.priority = PM_PRIO_LEVEL_0;
    pm_reg.flags = 0;
    pm_reg.chid = himax->thread_chid;

    /* Register himax with Power Manager */
    ret = devctl(himax->pm_fd, DCMD_PM_REGISTER, &pm_reg, sizeof(struct pm_register_s), NULL);
    if (ret != EOK) {
        mtouch_error(himax->log_name, "devctl() failed: %s\n", strerror(errno));
        error_memory("Himax_Touch: devctl() failed: %s\n", strerror(errno));
        goto fail_pm;
    }
    mtouch_info(himax->log_name, "STR registration successful %d\n", ret);

    /* Success */
    return 0;

fail_pm:
    close(himax->pm_fd);
fail:
    return -1;
}

int himax_open_i2c_device(himax_dev_t *himax)
{
    int retval;

    himax->i2c_fd = i2c_open(himax->i2c_devname);
    if (himax->i2c_fd == -1) {
        mtouch_error(himax->log_name, "Fail to open I2C device %s", himax->i2c_devname);
        error_memory("Himax_Touch: Fail to open I2C device %s", himax->i2c_devname);
        retval = -ENODEV;
        goto exit;
    }

    if((i2c_set_slave_addr(himax->i2c_fd, himax->i2c_slave_addr, I2C_ADDRFMT_7BIT)) == -1) {
        mtouch_error(himax->log_name, "Fail to set mtouch slave address %s", himax->i2c_devname);
        error_memory("Himax_Touch: Fail to set mtouch slave address %s", himax->i2c_devname);
        retval = -ENODEV;
        goto exit;
    }

    if((i2c_set_bus_speed(himax->i2c_fd, I2C_SPEED_HIGH, &himax->i2c_speed)) == -1)
        mtouch_debug(himax->log_name, "Failed to set i2c speed, default speed will be set"); /* default speed is set,even on failure */

    mtouch_info(himax->log_name, "I2C registration success, Device: %s, address: 0x%02x",
                 himax->i2c_devname, himax->i2c_slave_addr);

    retval = himax->i2c_fd;

exit:
    return retval;
}

static int attach_driver(himax_dev_t* himax)
{
    mtouch_driver_funcs_t funcs = {
        .get_contact_id = get_contact_id,
        .is_contact_down = is_contact_down,
        .get_coords = get_coords,
        .get_down_count = NULL,
        .get_touch_width = NULL,
        .get_touch_height = NULL,
        .get_touch_orientation = NULL,
        .get_touch_pressure = get_touch_pressure,
        .get_seq_id = NULL,
        .set_event_rate = NULL,
        .get_contact_type = NULL,
        .get_select = NULL
    };

    mtouch_driver_params_t params = {
        .capabilities = MTOUCH_CAPABILITIES_CONTACT_ID |
            MTOUCH_CAPABILITIES_COORDS |
            MTOUCH_CAPABILITIES_PRESSURE ,

        .flags = 0,
        .max_touchpoints = himax->max_touchpoints,
        .width = himax->width,
        .height = himax->height
    };
    himax->inputevents_hdl = mtouch_driver_attach(&params, &funcs);

    if (NULL == himax->inputevents_hdl) {
        mtouch_error(himax->log_name, "Failed to connect to libinputevents\n");
        error_memory("Himax_Touch: Failed to connect to libinputevents\n");
        return ERROR;
    }
    return EOK;
}

static int himax_hw_reset(himax_dev_t *himax)
{
    uint8_t  ret, len = 2;
    uint8_t buf[2] = {0};
    int fd, Ctrl_handle, RetVal = 0;

    Ctrl_handle = displaybinder_ctrl_open(himax->dev_ctrl_path);  // dev_control_path -> dev/vcd/display-binder/control
    if(Ctrl_handle < 0)
    {
        mtouch_error(himax->log_name, "Error to open display binder control path");
        error_memory("Himax_Touch: Error to open display binder control path");
    } else {
       RetVal = displaybinder_ctrl_set(Ctrl_handle, "touch-reset","one-time");
       if(RetVal < 0)
       {
           mtouch_error(himax->log_name, "Failed to reset the touch controller through FIDM API");
           error_memory("Himax_Touch: Failed to reset the touch controller through FIDM API");
           displaybinder_ctrl_close(Ctrl_handle);
       } else {
             mtouch_info(himax->log_name, "executed touch reset successfully through FIDM API\n");
             displaybinder_ctrl_close(Ctrl_handle);
             return RetVal;
       }
    }
    if((Ctrl_handle < 0) || (RetVal < 0))
    {

        fd = i2c_open(himax->i2c_devname);
        if (fd == -1)
            return -ENODEV;

        if((i2c_set_slave_addr(fd, 0x12, I2C_ADDRFMT_7BIT)) == -1) {
            mtouch_error(himax->log_name, "Fail to set mcu slave address %s", himax->i2c_devname);
            error_memory("Himax_Touch: Fail to set mcu slave address %s", himax->i2c_devname);
            i2c_close(fd);
            return -1;
        }

        if((i2c_set_bus_speed(fd, I2C_SPEED_HIGH, &himax->i2c_speed)) == -1) {
            mtouch_debug(himax->log_name, "Failed to set i2c speed for mcu, default speed will be set"); /* default speed is set,even on failure */
        }

        buf[0] = 0x31;
        buf[1] = 0x01;

        ret = i2c_write(fd, buf, len);
        i2c_close(fd);

        if(ret == len) {
            return 0;
        }
   }
    return -1;
}

static int himax_init(himax_dev_t *himax)
{
    int i;
    uint8_t tmp_data[4];
    uint8_t tmp_addr[4];
    int ret = EOK;

    mtouch_error(himax->log_name, "Resetting.....");
    ret = himax_hw_reset(himax);
    if (ret != EOK) {
        mtouch_error(himax->log_name, "Failed to execute hard reset of touch controller %d\n", ret);
        error_memory("Himax_Touch: Failed to execute hard reset of touch controller %d\n", ret);
         return ENODEV;
    }
    if (usleep(500*1000) != 0) {
        mtouch_error(himax->log_name, "Failed to execute 500ms sleep %d\n", errno);
        error_memory("Himax_Touch: Failed to execute 500ms sleep %d\n", errno);
    }

    mtouch_info(himax->log_name, "Executed hard resest successfully\n");
    //usleep(100*000);
    tmp_addr[0] = 0xD0;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x00;
    tmp_addr[3] = 0x90;
    himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);
    mtouch_info(himax->log_name, "IC_ID : HX_%02X%02X%02X\n", tmp_data[3], tmp_data[2], tmp_data[1]);

    ret = himax_mcu_read_FW_ver(himax);
    if (ret != EOK) {
        mtouch_error(himax->log_name, "Fail to read FW version \n");
        error_memory("Himax_Touch: Fail to read FW version \n");
        return ret;
    }

    for (i = 0; i< MAX_TOUCH; i++) {
        himax->pre_finger_state[i] = CONTACT_UP;
        himax->prev_touch_state[i] = CONTACT_UP;
    }


    return EOK;
}

int himax_mtouch_suspend(himax_dev_t *himax)
{
    int MaskCnt;

    MaskCnt = InterruptMask(himax->tp_intr, himax->tp_iid);
    mtouch_info(himax->log_name, "%s-%d>MaskCnt[%d]", __FUNCTION__,__LINE__,MaskCnt);
    himax_release_touch(himax);

    return 0;
}

int himax_mtouch_resume(himax_dev_t *himax)
{
    int ret;
    int retry_count = 0;
    int MaskCnt = 0;
    int hw_init_flag = 0;

retry_init:
    ret = himax_init(himax);
    if (ret != 0) {
       if (retry_count < MAX_RETRY_ATTEMPTS) {
          retry_count++;
          mtouch_info(himax->log_name, "HW initialization failed, retrying %d\n",retry_count);
          if (usleep(500*1000) != 0) {
             mtouch_error(himax->log_name, "Failed to execute 500ms sleep %d\n", errno);
             error_memory("Himax_Touch: Failed to execute 500ms sleep %d\n", errno);
          }
          goto retry_init;
       } else {
          mtouch_error(himax->log_name, "HW initialization failed %d", ret);
          error_memory("Himax_Touch: HW initialization failed %d", ret);
          return -1;
       }
    }
    else
    {
        if(himax->HW_Init_status == false)
        {
          hw_init_flag = 1;
          /*unmasking the interrupt masked in himax_create_isr*/
          MaskCnt = InterruptUnmask(himax->tp_intr, himax->tp_iid);
          mtouch_info(himax->log_name,"%s-%d>UnMaskCnt[%d]", __FUNCTION__,__LINE__,MaskCnt);
        }
    }
    /*unmasking the interrupt masked in SUSPEND*/
    if((hw_init_flag==0) || (MaskCnt > 0 ))
    {
         MaskCnt = InterruptUnmask(himax->tp_intr, himax->tp_iid);
         mtouch_info(himax->log_name,"%s-%d>UnMaskCnt[%d]", __FUNCTION__,__LINE__,MaskCnt);
	}

    return 0;
}

void himax_log_touch_data(uint8_t *buf,uint8_t touch_info_size)
{
    int loop_i = 0;
    int print_size = 0;
    print_size = touch_info_size;

    for (loop_i = 0; loop_i < print_size; loop_i += 8) {
        if((loop_i + 7) >= print_size) {
            mtouch_error("Himax", "over flow\n");
            error_memory("Himax_Touch: over flow\n");
            break;
        }
        mtouch_info("Himax", "P %2d = 0x%2.2X P %2d = 0x%2.2X ", loop_i, buf[loop_i], loop_i + 1, buf[loop_i + 1]);
        mtouch_info("Himax", "P %2d = 0x%2.2X P %2d = 0x%2.2X ", loop_i + 2, buf[loop_i + 2], loop_i + 3, buf[loop_i + 3]);
        mtouch_info("Himax", "P %2d = 0x%2.2X P %2d = 0x%2.2X ", loop_i + 4, buf[loop_i + 4], loop_i + 5, buf[loop_i + 5]);
        mtouch_info("Himax", "P %2d = 0x%2.2X P %2d = 0x%2.2X ", loop_i + 6, buf[loop_i + 6], loop_i + 7, buf[loop_i + 7]);
        mtouch_info("Himax", "\n");
    }
}

int himax_read_event_stack(himax_dev_t *himax, uint8_t *read_data, uint8_t read_len)
{
    if(himax_i2c_combined_writeread(himax->i2c_fd, H_CMD_EVENT_STACK, read_data, read_len) != EOK) {
        mtouch_error(himax->log_name, "himax_adap_combined_writeread i2c access fail!\n");
        return EIO;
    }
    //mtouch_info(HIMAX_DEVNAME, "End\n");
    return EOK;
}

static void himax_report_points(himax_dev_t *himax)
{
    int x = 0;
    int y = 0;
    int w = 0;
    int base = 0;
    int32_t    loop_i = 0;


    /* finger on/press */
    if (himax->hx_point_num != 0 ) {
        for (loop_i = 0; loop_i < MAX_TOUCH; loop_i++) {
            base = loop_i * 4;
            x = (himax->hx_coord_buf[base] << 8 | himax->hx_coord_buf[base + 1]);
            y = (himax->hx_coord_buf[base + 2] << 8 | himax->hx_coord_buf[base + 3]);
            w = himax->hx_coord_buf[(MAX_TOUCH * 4) + loop_i];

            if (himax->verbose > 6) {
                 mtouch_info(himax->log_name, "Coords status=>F:%02d Down, X:%d, Y:%d, P:%d\n",loop_i+1, x, y, w);
            }
            /* finger on/press */
            if(x != 0xFFFF && y != 0xFFFF) {
                switch (himax->pre_finger_state[loop_i]) {
                case CONTACT_DOWN:
                    himax->pre_finger_state[loop_i] = CONTACT_MOVE;
                    himax->state[loop_i].contact = CONTACT_MOVE;
                    break;
                case CONTACT_UP:
                    himax->pre_finger_state[loop_i] = CONTACT_DOWN;
                    himax->state[loop_i].contact = CONTACT_DOWN;
                    break;
                case CONTACT_MOVE:
                    himax->state[loop_i].contact = CONTACT_MOVE;
                    break;
                default:
                    break;
                }
                himax->state[loop_i].pressure = w;
            /* finger up */
            } else {
                himax->pre_finger_state[loop_i] = CONTACT_UP;
                himax->state[loop_i].contact = CONTACT_UP;
                himax->state[loop_i].pressure = 0;
            }
            himax->state[loop_i].x = x;
            himax->state[loop_i].y = y;
            himax->state[loop_i].t = loop_i;
            himax->state[loop_i].is_valid = 1;
        }

    /* finger leave/release */
    } else {
        for (loop_i = 0; loop_i < MAX_TOUCH; loop_i++) {

            himax->state[loop_i].pressure = 0;
            himax->state[loop_i].is_valid = 1;
            himax->state[loop_i].contact = CONTACT_UP;
        }
        if (himax->verbose > 6) {
            mtouch_info(himax->log_name, "Finger all leave\n");
        }
    }

    if (himax->verbose > 6) {
        mtouch_info(himax->log_name, "End\n");
    }
}

void himax_coord_report(himax_dev_t *himax)
{

    if (himax->hx_coord_buf[TOUCH_INFO_POINT_CNT] == 0xFF)
        himax->hx_point_num = 0;
    else
        himax->hx_point_num= himax->hx_coord_buf[TOUCH_INFO_POINT_CNT] & 0x0F;

    himax_report_points(himax);

}

int himax_checksum_cal(uint8_t *buf)
{
    uint16_t check_sum_cal = 0;
    int32_t    loop_i = 0;

    for (loop_i = 0; loop_i < TOUCH_INFO_SIZE; loop_i++) {
        check_sum_cal += buf[loop_i];
    }

    if ((check_sum_cal % 0x100 != 0) ) {
        mtouch_error("Himax", "[HIMAX TP MSG] checksum fail : check_sum_cal: 0x%02X\n", check_sum_cal);
        return CHECKSUM_FAIL;
    }

    //mtouch_error(HIMAX_DEVNAME, "End\n");

    return EOK;
}

int himax_get_touch(himax_dev_t *himax)
{
    uint8_t buf[TOUCH_INFO_SIZE * 2];
    uint16_t check_sum_cal = 0;

    memset(buf, 0x00, sizeof(buf));
    if(himax_read_event_stack(himax, buf, TOUCH_INFO_SIZE) != EOK) {
        mtouch_error(himax->log_name, "can't read data from chip!\n");
        error_memory("Himax_Touch: can't read data from chip!\n");
        return EIO;
    }

    if (himax->verbose > 6) {
        himax_log_touch_data(buf, TOUCH_INFO_SIZE);
    }

    check_sum_cal = himax_checksum_cal(buf);
    if (check_sum_cal != EOK) {
        mtouch_error(himax->log_name, "error calculating checksum %d\n", check_sum_cal);
        return CHECKSUM_FAIL;
    }
    else
        memcpy(himax->hx_coord_buf, &buf[0], TOUCH_INFO_SIZE);

    himax_coord_report(himax);

    return EOK;

}

void himax_dump_rtd_record(himax_dev_t *himax, uint8_t *data)
{
    char *buf;
    int i;

    buf = (char *) malloc (MAX_RTD_DATA_LENGTH);
    if (buf == NULL) {
        mtouch_error (himax->log_name, "Failed to allocate buffer space to dump touch rtd record");
        error_memory("Himax_Touch: Failed to allocate buffer space to dump touch rtd record");
        return;
    }

    sprintf (buf, "%x ", data[0]);

    for (i = 1; i < MAX_RTD_DATA_LENGTH; i++) {
        sprintf ((buf + strlen(buf)), "%x ", data[i]);
    }

    mtouch_debug(himax->log_name,"touch rtd counter = 0x%x", data[MAX_RTD_DATA_LENGTH - 2]);
    if (himax->verbose > 7)
        mtouch_debug(himax->log_name,"touch rtd data = %s", buf);

    free(buf);

}

void himax_dump_knob_rtd_record(himax_dev_t *himax, uint8_t *data)
{
    char *buf;
    int i;

    buf = (char *) malloc (MAX_KNOB_RTD_DATA_LENGTH);
    if (buf == NULL) {
        mtouch_error (himax->log_name, "Failed to allocate buffer space to dump knob rtd record");
        error_memory("Himax_Touch: Failed to allocate buffer space to dump knob rtd record");
        return;
    }
    sprintf (buf, "%x ", data[KNOB_RTD_DATA]);
    for (i = KNOB_RTD_DATA+1; i < MAX_KNOB_RTD_DATA_LENGTH + KNOB_RTD_DATA ; i++) {
        sprintf ((buf + strlen(buf)), "%x ", data[i]);
    }

    mtouch_debug(himax->log_name,"knob rtd counter = 0x%x%x", data[KNOB_RTD_DATA], data[KNOB_RTD_DATA+1]);
    if (himax->verbose > 7)
        mtouch_debug(himax->log_name,"knob rtd data = %s", buf);

    free(buf);

}

int himax_get_display_grpid(himax_dev_t *himax)
{
    int handle, ret;
    char *reply;

    reply = NULL;

    if(!himax->dev_status_path)
    {
        mtouch_error(himax->log_name,"Not a valid status path %s", himax->dev_status_path);
        error_memory("Himax_Touch: Not a valid status path %s", himax->dev_status_path);
        return -EINVAL;
    }
    handle = displaybinder_status_open(himax->dev_status_path);
    if(handle < 0)
    {
        mtouch_error(himax->log_name, "Failed to open display binder -Dev status path: %s, handle:%d", himax->dev_status_path, handle);
        error_memory("Himax_Touch: Failed to open display binder -Dev status path: %s, handle:%d", himax->dev_status_path, handle);
        return handle;
    }

    mtouch_debug(himax->log_name,"display binder open success(reval: %d) for dev status path %s", handle, himax->dev_status_path);
    ret = displaybinder_status_get(handle,"display-group-id",&reply);
    displaybinder_status_close(handle);
    if (ret)
    {
        mtouch_error(himax->log_name,"failed to fetch group id: %d", ret);
        error_memory("Himax_Touch: failed to fetch group id: %d", ret);
    }
    else {
        himax->display_group_id = atoi(reply);
        mtouch_info (himax->log_name, "Display Group ID is fetched successfully: %d", himax->display_group_id);
    }

    if(reply)
       free(reply);

    return ret;
}

int himax_process_rtd(himax_dev_t *himax)
{

    int ret;

    if(himax->display_group_id < 0)
    {
       ret = himax_get_display_grpid(himax);
        if(ret != 0)
        {
           mtouch_error(himax->log_name,"failed to fetch group id: %d again", himax->display_group_id);
           error_memory("Himax_Touch: failed to fetch group id: %d again", himax->display_group_id);
           return -1;
        }
    }

    memcpy(himax->rtd_data, &himax->hx_coord_buf[RTD_DATA], MAX_RTD_DATA_LENGTH-1);
    himax->rtd_data[MAX_RTD_DATA_LENGTH-1] = himax->display_group_id;
    himax->rtd_len = MAX_RTD_DATA_LENGTH;

    if (himax->verbose > 6) {
        himax_dump_rtd_record(himax, himax->rtd_data);
    }

    memcpy(himax->knob_rtd, &himax->hx_coord_buf[KNOB_RTD_DATA], MAX_KNOB_RTD_DATA_LENGTH);
    himax->knob_rtd_len = MAX_KNOB_RTD_DATA_LENGTH;

    if (himax->verbose > 6) {
        himax_dump_knob_rtd_record(himax, himax->hx_coord_buf);
    }

    return EOK;

}

static int himax_connect_faceplate(himax_dev_t *himax)
{
    char *attach_point;

    attach_point = strdup(FACEPLATE_ATTACH_POINT);
    himax->faceplate_coid = name_open(attach_point, 0);
    if (himax->faceplate_coid == -1)
    {
        mtouch_error(himax->log_name, "Failed to connect to faceplate driver, coid: %d, %d", himax->faceplate_coid, errno);
        error_memory("Himax_Touch: Failed to connect to faceplate driver, coid: %d, %d", himax->faceplate_coid, errno);
        return -1;
    }
    mtouch_info(himax->log_name, "Connected to faceplate driver successfully");
    return 0;
}

int himax_send_knob_data(himax_dev_t *himax)
{
    int ret;
    faceplate_msg msg;

    if (himax->faceplate_coid == -1) {
        if ((ret = himax_connect_faceplate(himax)) < 0)
            return -EAGAIN;
    }

    msg.message.command = FACEPLATE_KNOB_SUPPORT;
    msg.message.buff = himax->knob_data;

    ret = MsgSend(himax->faceplate_coid, &msg, sizeof(msg), NULL, 0);
    if (ret != EOK) {
        return ret;
    }
    else {
        if (himax->verbose > 6) {
            mtouch_info(himax->log_name, "Knob data sent, ret = %d data = %d\n", ret, himax->knob_data);
        }
        return ret;
    }
}

/* read xy_data for all current touches */
int himax_xy_worker(himax_dev_t *himax)
{
    int ret, i;
    bool knob_data_flag = false;
    ret = himax_get_touch(himax);
    if(ret != EOK){
        mtouch_error(himax->log_name, "error processing touch data  ret = %d errno = %d\n", ret, errno);
        return ret;
    }
    if(himax->hx_coord_buf[0] == 0xDA && himax->hx_coord_buf[1] == 0xDA && himax->hx_coord_buf[2] == 0xDA && himax->hx_coord_buf[3] == 0xDA ) {
#ifdef BOSCH_RTC_2574268_HID_ENABLE_HIMAX_LOGGING_IMPROVEMENTS
         // Increment and log RTD update received counter
         himax->rtd_update_recvd_counter++;
         if (himax->rtd_update_recvd_counter >= UINT32_MAX) {
             mtouch_warn(himax->log_name, "rtd_update_recvd_counter reached maximum value, resetting to 0");
             himax->rtd_update_recvd_counter = 0;
         }
         if (himax->rtd_update_recvd_counter % LOG_RTD_PROCESS_COUNT == 0) {
             mtouch_info(himax->log_name, "RTD UPDATE RECEIVED COUNTER: %d", himax->rtd_update_recvd_counter);
         }
#endif
         himax_process_rtd(himax);
         himax->rtd_readflag = 1;
         himax->knob_rtd_readflag = 1;

#ifdef BOSCH_RTC_2574268_HID_ENABLE_HIMAX_LOGGING_IMPROVEMENTS
         himax->rtd_intr_flag = 1;
         // Increment and log RTD update processed counter
         himax->rtd_update_prcsd_counter++;
         if (himax->rtd_update_prcsd_counter >= UINT32_MAX) {
             mtouch_warn(himax->log_name, "rtd_update_prcsd_counter reached maximum value, resetting to 0");
             himax->rtd_update_prcsd_counter = 0;
         }
         if (himax->rtd_update_prcsd_counter % LOG_RTD_PROCESS_COUNT == 0) {
             mtouch_info(himax->log_name, "RTD UPDATE PROCESSED COUNTER: %d", himax->rtd_update_prcsd_counter);
         }
#endif
    }
    else {
        mtouch_driver_process_packet(himax->inputevents_hdl, himax->state, himax, MTOUCH_PARSER_FLAG_NONE);

        for(i = 0; i < 55; i++) {
            if (himax->hx_coord_buf[i] != 0xFF) {
                knob_data_flag = true;
            }
        }

        if (knob_data_flag) {
            if (himax->hx_coord_buf[KNOB_DATA]) {
                himax->knob_data = (int8_t) himax->hx_coord_buf[KNOB_DATA];
                mtouch_debug(himax->log_name, "knob data %d\n", himax->knob_data);
                ret = himax_send_knob_data(himax);
                if (ret != EOK) {
                    mtouch_error(himax->log_name, "Failed to send knob data to faceplate driver  ret = %d errno = %d\n", ret, errno);
                    error_memory("Himax_Touch: Failed to send knob data to faceplate driver  ret = %d errno = %d\n", ret, errno);
                }
            }
        }
    }

    return ret;
}

void *himax_tp_recv_thread(void* arg)
{
    himax_dev_t *himax = arg;
    int rcvid, ret;
    iov_t iov;
    struct _pulse pulse;
    struct pm_ack_s ack = { .rc = 0 };

    /* prepare for the simple messages passing
     * fill in the fields of an iov_t structure
     */
    SETIOV (&iov, &pulse, sizeof(pulse));

    if (ThreadCtl(_NTO_TCTL_IO, 0) == -1) {
        mtouch_error(himax->log_name, "Fail to config ThreadCtl");
        error_memory("Himax_Touch: Fail to config ThreadCtl");
        return 0;
    }

    while (1) {
        rcvid = MsgReceivev(himax->thread_chid, &iov, 1, NULL);
        if (rcvid < 0) {
            if ((EINTR == errno) || (ETIMEDOUT == errno)) {
                continue;
            }
            mtouch_error(himax->log_name, "MsgReceive failed: %s", strerror(errno));
            error_memory("Himax_Touch: MsgReceive failed: %s", strerror(errno));
            goto fail;
        } else if (rcvid == 0) {
            /* Pulse received */
            switch (pulse.code) {
                case HIMAX_PULSE_CODE:
#ifdef BOSCH_RTC_2574268_HID_ENABLE_HIMAX_LOGGING_IMPROVEMENTS
                himax->Tp_pulse_received_counter++;
                 // Reset the counter if it reaches the maximum value to prevent overflow
                if (himax->Tp_pulse_received_counter >= UINT32_MAX) {
                    mtouch_warn(himax->log_name, "Tp_pulse_received_counter reached maximum value, resetting to 0");
                    himax->Tp_pulse_received_counter = 0;
                }
                /* Print the counter every 2 secs */
                if(himax->Tp_pulse_received_counter % LOG_RTD_REQUEST_COUNT == 0)
                {
                   mtouch_info (himax->log_name, "Tp_pulse_received_counter : %d", himax->Tp_pulse_received_counter);
                }
#else
                if ( himax->verbose > 5 ) {
                    mtouch_info(himax->log_name, "Received an interrupt (pulse) from the controller");
                }
#endif
                if (himax->inputevents_hdl != NULL) {
                  ret = himax_xy_worker(himax);
                  if (ret != 0) {
                    mtouch_error(himax->log_name,"failed to process touch messages\n");
                  }
                }
                else {
                    if ( himax->verbose > 5 )
                         mtouch_error(himax->log_name,"mtouch_interrupt_handling_thread: himax->inputevents_hdlr is Null\n");
                }
                    InterruptUnmask(himax->tp_intr, himax->tp_iid);
#ifdef BOSCH_RTC_2574268_HID_ENABLE_HIMAX_LOGGING_IMPROVEMENTS
                    himax->Tp_pulse_unmasked_counter++;
                    // Reset the counter if it reaches the maximum value to prevent overflow
                if (himax->Tp_pulse_unmasked_counter >= UINT32_MAX) {
                    mtouch_warn(himax->log_name, "Tp_pulse_unmasked_counter reached maximum value, resetting to 0");
                    himax->Tp_pulse_unmasked_counter = 0;
                }
                if(himax->rtd_intr_flag == 1)
                {
                   himax->Tp_pulse_unmasked_rtd_counter++;
                   // Reset the counter if it reaches the maximum value to prevent overflow
                   if (himax->Tp_pulse_unmasked_rtd_counter >= UINT32_MAX) {
                       mtouch_warn(himax->log_name, "Tp_pulse_unmasked_rtd_counter reached maximum value, resetting to 0");
                       himax->Tp_pulse_unmasked_rtd_counter = 0;
                   }
                   himax->rtd_intr_flag = 0;
                }
                if(himax->Tp_pulse_unmasked_counter % LOG_RTD_REQUEST_COUNT == 0)
                {
                   mtouch_info (himax->log_name, "Tp_pulse_unmasked_counter : %d", himax->Tp_pulse_unmasked_counter);
                   mtouch_info (himax->log_name, "Tp_pulse_unmasked_rtd_counter : %d", himax->Tp_pulse_unmasked_rtd_counter);
                }
#endif
                    break;
                case HIMAX_PM_PREPARE_PULSE:
                    mtouch_info(himax->log_name, "Received HIMAX_PM_PREPARE_PULSE");
                    if(himax->himax_pm_state == PM_STATE_SUSPEND)
                    {
                        mtouch_info(himax->log_name, "Ignoring the HIMAX_PM_PREPARE_PULSE since mtouch driver is already in PREPARE state");

                        /* Send ACK to Power Manager */
                        ack.rc = 0;
                    } else {
                        /* prepare himax suspend */
                        // do nothing

                        if (ack.rc) {
                            mtouch_error(himax->log_name, "Failed to handle HIMAX_PM_PREPARE_PULSE");
                            error_memory("Himax_Touch: Failed to handle HIMAX_PM_PREPARE_PULSE");
                        } else {
                            /* Update himax PM state */
                            himax->himax_pm_state = PM_STATE_PREPARE;
                        }
                    }
                    /* send ACK to power manager */
                    ack.state = PM_STATE_PREPARE;
                    ret = devctl(himax->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(himax->log_name, "devctl(DCMD_PM_ACK) for HIMAX_PM_PREPARE_PULSE failed: %s\n", strerror(errno));
                        error_memory("Himax_Touch: devctl(DCMD_PM_ACK) for HIMAX_PM_PREPARE_PULSE failed: %s\n", strerror(errno));
                    }
                    break;
                case HIMAX_PM_SUSPEND_PULSE:
                    mtouch_info(himax->log_name, "Received HIMAX_PM_SUSPEND_PULSE");
                    if(himax->himax_pm_state == PM_STATE_SUSPEND)
                    {
                        mtouch_info(himax->log_name, "Ignoring the HIMAX_PM_SUSPEND_PULSE since mtouch driver is already in SUSPEND state");

                        /* Send ACK to Power Manager */
                        ack.rc = 0;
                    } else {
                        /* process suspend */
                        ack.rc = himax_mtouch_suspend(himax);

                        if (ack.rc) {
                            mtouch_error(himax->log_name, "Failed to handle HIMAX_PM_SUSPEND_PULSE");
                            error_memory("Himax_Touch: Failed to handle HIMAX_PM_SUSPEND_PULSE");
                        } else {
                        /* Update PM state */
                        himax->himax_pm_state = PM_STATE_SUSPEND;
                        }
                    }
                    /* Send ACK to Power Manager */
                    ack.state = PM_STATE_SUSPEND;
                    ret = devctl(himax->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(himax->log_name, "devctl(DCMD_PM_ACK) for HIMAX_PM_SUSPEND_PULSE failed: %s\n", strerror(errno));
                        error_memory("Himax_Touch: devctl(DCMD_PM_ACK) for HIMAX_PM_SUSPEND_PULSE failed: %s\n", strerror(errno));
                    }
                    break;
                case HIMAX_PM_RESUME_PULSE:
                    mtouch_info(himax->log_name, "Received HIMAX_PM_RESUME_PULSE");
                    ack.state = PM_STATE_RESUME;
                    ret = devctl(himax->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(himax->log_name, "devctl(DCMD_PM_ACK) for HIMAX_PM_RESUME_PULSE: failed: %s\n", strerror(errno));
                        error_memory("Himax_Touch: devctl(DCMD_PM_ACK) for HIMAX_PM_RESUME_PULSE: failed: %s\n", strerror(errno));
                    } else {
                        mtouch_info(himax->log_name, "Sending Ack to power manager about RESUME PULSE received");
                    }
                    /* process himax resume */
                    if(himax->himax_pm_state == PM_STATE_RESUME)
                    {
                        mtouch_info(himax->log_name, "Ignoring the HIMAX_PM_RESUME_PULSE since mtouch driver is already in RESUME state");
                        ack.rc = 0;
                    }
                    else {
                        ack.rc = himax_mtouch_resume(himax);
                        /* Send ACK to Power Manager */
                        if (ack.rc) {
                            mtouch_error(himax->log_name, "Failed to handle HIMAX_PM_RESUME_PULSE");
                            error_memory("Himax_Touch: Failed to handle HIMAX_PM_RESUME_PULSE");
                        } else {
                            /* Update himax PM state */
                            himax->himax_pm_state = PM_STATE_RESUME;
                        }
                    }
                    break;
                default:
                    mtouch_warn(himax->log_name, "Received unknown pulse: %d", pulse.code);
                    break;
            }
        }
    }

fail:
    mtouch_critical(himax->log_name, "Failed to to configure TP\n");
    return NULL;
}

void *himax_ext_msg_handler(void *arg)
{
    himax_dev_t* himax = arg;
    fidm_touch_msg_u msg;
    int rcvid;
    int ret;

    while (1) {
        rcvid = MsgReceive(himax->fidm_attach->chid, &msg, sizeof(msg), NULL);
        if (rcvid < 0) {
            if ((EINTR == errno) || (ETIMEDOUT == errno)) {
                continue;
            }
            mtouch_error(himax->log_name, "MsgReceive failed: %s %d", strerror(errno), rcvid);
            error_memory("Himax_Touch: MsgReceive failed: %s %d", strerror(errno), rcvid);
            goto fail;
        } else if (rcvid != 0) {
            /* if himax is in suspend/prepare state, return EAGAIN to try after sometime */
            if ((himax->himax_pm_state != PM_STATE_RESUME) && (msg.type != FIDM_TOUCH_GET_GROUPID)) {
                if (himax->verbose > 5)
                    mtouch_error(himax->log_name, "Send EAGAIN: pm state: %d, msg type: %d", himax->himax_pm_state, msg.type);

                MsgError(rcvid, -EAGAIN);
                continue;
            }
            /* msg received */
            switch(msg.type) {
                case FIDM_TOUCH_GET_RTD_DATA:
                    if (himax->verbose > 5) {
                        mtouch_info(himax->log_name, "Received FIDM_TD_MSG_GET_RTD_DATA");
                    }

#ifdef BOSCH_RTC_2574268_HID_ENABLE_HIMAX_LOGGING_IMPROVEMENTS
                    himax->fidm_RTD_rcvd_request_counter++;

                    // Check if the counter exceeds the maximum value and reset if necessary
                    if (himax->fidm_RTD_rcvd_request_counter == INT_MAX) {
                        himax->fidm_RTD_rcvd_request_counter = 0;
                        mtouch_info(himax->log_name, "RTD received request counter reset to 0 after reaching INT_MAX");
                    }

                    if (himax->rtd_readflag == 1) {
                        ret = MsgReply(rcvid, 0, himax->rtd_data, himax->rtd_len);
                        if (ret == EOK) {
                            // Increment processed request counter and reset read flag
                            himax->fidm_RTD_prcsd_request_counter =
                                (himax->fidm_RTD_prcsd_request_counter + 1) % INT_MAX;
                            himax->rtd_readflag = 0;
                        }
#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
                        else if (errno == ESRCH) {
                            // Handle ESRCH error specifically
                            mtouch_error(himax->log_name,"MsgReply failed: ESRCH - No such process");
                            MsgError(rcvid, -errno);
                        }
#endif
                        else {
                            // Handle error in MsgReply
                            mtouch_info(himax->log_name, "Msg replied error with rtd data: %d", errno);
                            MsgError(rcvid, -errno);
                        }

                        if (himax->verbose > 5) {
                            mtouch_info(himax->log_name, "Msg replied with touch rtd data: %d", errno);
                        }
                    } else {
                        MsgError(rcvid, -EAGAIN);
                        himax->fidm_RTD_rcvd_request_error_counter =
                            (himax->fidm_RTD_rcvd_request_error_counter + 1) % INT_MAX;
                    }

                    if (himax->fidm_RTD_rcvd_request_counter % LOG_RTD_REQUEST_COUNT == 0) {
                        mtouch_info(himax->log_name, "FIDM RTD received request counter: %d",
                                    himax->fidm_RTD_rcvd_request_counter);
                    }

                    if ((himax->fidm_RTD_prcsd_request_counter % LOG_RTD_REQUEST_COUNT == 0) &&
                        (himax->fidm_RTD_prcsd_bkup_counter != himax->fidm_RTD_prcsd_request_counter)) {
                        himax->fidm_RTD_prcsd_bkup_counter = himax->fidm_RTD_prcsd_request_counter;

                        if (himax->rtd_len >= 42) { // Ensure array bounds are not exceeded
                            mtouch_info(himax->log_name, "RTD= %x %x with length = %d",
                                        himax->rtd_data[40], himax->rtd_data[41], himax->rtd_len);
                        } else {
                            mtouch_warn(himax->log_name, "RTD data length insufficient for logging: %d",
                                        himax->rtd_len);
                        }

                        mtouch_info(himax->log_name,
                                    "FIDM RTD processed request counter: %d, received error counter: %d",
                                    himax->fidm_RTD_prcsd_request_counter,
                                    himax->fidm_RTD_rcvd_request_error_counter);
                    }
#else
                    if (himax->rtd_readflag == 1) {
                        ret = MsgReply(rcvid, 0, himax->rtd_data, himax->rtd_len);
                        if (ret == EOK) {
                            himax->rtd_readflag = 0;
                        }
#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
                        else if (errno == ESRCH) {
                            // Handle ESRCH error specifically
                            mtouch_error(himax->log_name,"MsgReply failed: ESRCH - No such process");
                            MsgError(rcvid, -errno);
                        }
#endif
                        else {
                            MsgError(rcvid, -errno);
                        }

                        if (himax->verbose > 5) {
                            mtouch_info(himax->log_name, "Msg replied with touch rtd data: %d", errno);
                        }
                    } else {
                        MsgError(rcvid, -EAGAIN);
                    }
#endif
                    break;
                case FIDM_TOUCH_GET_GROUPID:
                    mtouch_info(himax->log_name, "Received FIDM_TOUCH_GET_GROUPID");
                    if (himax->display_group_id < 0) {
                        ret = himax_get_display_grpid(himax);
                        if (ret != 0)
                        {
                            mtouch_error(himax->log_name,"failed to fetch group id: %d again", himax->display_group_id);
                            error_memory("Himax_Touch: failed to fetch group id: %d again", himax->display_group_id);
                            MsgError(rcvid, -errno);
                        }
                        else {
                            ret = MsgReply(rcvid, 0, &himax->display_group_id, sizeof(himax->display_group_id));
                            if (ret != EOK) {
                                MsgError(rcvid, -errno);
                            }
                        }
                    }
                    else {
                        ret = MsgReply(rcvid, 0, &himax->display_group_id, sizeof(himax->display_group_id));
                        mtouch_info(himax->log_name, "GROUPID %d ", himax->display_group_id);
                        if (ret != EOK) {
                            MsgError(rcvid, -errno);
                        }
                    }
                    break;
                case FIDM_TOUCH_GET_KNOB_RTD_DATA:
                    if (himax->verbose > 5)
                        mtouch_info(himax->log_name, "Received FIDM_TOUCH_GET_KNOB_RTD_DATA");

                    if (himax->knob_rtd_readflag == 1) {
                        ret = MsgReply(rcvid, 0, himax->knob_rtd, himax->knob_rtd_len);
                        if (ret == EOK) {
                            himax->knob_rtd_readflag = 0;
                        }
#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
                        else if (errno == ESRCH) {
                            // Handle ESRCH error specifically
                            mtouch_error(himax->log_name,"MsgReply failed: ESRCH - No such process");
                            MsgError(rcvid, -errno);
                        }
#endif
                        else {
                            MsgError(rcvid, -errno);
                        }
                        if (himax->verbose > 5)
                            mtouch_info(himax->log_name, "Msg replied with knob rtd data: %d", errno);
                    }
                    else {
                        MsgError(rcvid, -EAGAIN);
                    }
                    break;
                default:
                    if (himax->verbose > 5)
                        mtouch_warn(himax->log_name, "%s: Invalid msg id received", __FUNCTION__);
                    MsgError(rcvid, -ENOTSUP);
                    break;
            }
        }
    }
fail:
    mtouch_critical(himax->log_name, "MsgReceive failed for external message handler");
    return NULL;
}
static int himax_create_isr(himax_dev_t *himax)
{
    int retval;
    int MaskCnt;

    if (!himax) {
        mtouch_error("Himax", "Invalid himax\n");
        error_memory("Himax_Touch: Invalid himax\n");
        return -EINVAL;
    }

    /* setup the interrupt communication channel */
    himax->thread_chid = ChannelCreate(_NTO_CHF_DISCONNECT | _NTO_CHF_UNBLOCK);
    if (himax->thread_chid < 0) {
        mtouch_error(himax->log_name, "Fail to create interrupt handling channel (error: %s)",
            strerror(errno));
        error_memory("Himax_Touch: Fail to create interrupt handling channel (error: %s)",
            strerror(errno));
        return -EIO;
    }

#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
    dispatch_t* dpp = dispatch_create_channel(himax->thread_chid, 0); // Create a dispatch structure for the channel
    if (dpp == NULL) {
            mtouch_error(himax->log_name,"%s: dispatch_create_channel failed: %s", __FUNCTION__, strerror(errno));
            error_memory("Himax_Touch: %s: dispatch_create_channel failed: %s", __FUNCTION__, strerror(errno));
            ChannelDestroy(himax->thread_chid); // Destroy the channel if dispatch creation fails
            return -EIO;
    }
#endif

    himax->thread_coid = ConnectAttach(0, 0, himax->thread_chid, _NTO_SIDE_CHANNEL, 0);
    if (himax->thread_coid < 0) {
        mtouch_error(himax->log_name, "Fail to attach interrupt handling channel (error: %s)",
            strerror(errno));
        error_memory("Himax_Touch: Fail to attach interrupt handling channel (error: %s)",
            strerror(errno));
        return -EIO;
    }

    /* initializes the thread attributes in defaults
     * defaults: PTHREAD_CANCEL_DEFERRED, PTHREAD_CANCEL_ENABLE, PTHREAD_CREATE_JOINABLE
     * PTHREAD_INHERIT_SCHED, PTHREAD_SCOPE_SYSTEM
     */
    pthread_attr_init(&himax->thread_attr);
    /* set the thread scheduling policy, round-robin scheduling. */
    pthread_attr_setschedpolicy(&himax->thread_attr, SCHED_RR);
    /* set a thread's scheduling parameters */
    himax->thread_param.sched_priority = himax->thread_priority;
    pthread_attr_setschedparam(&himax->thread_attr, &himax->thread_param);
    /* set the thread's inherit-scheduling attribute
     * use the scheduling policy specified in pthread_attr_t for the thread.
     */
    pthread_attr_setinheritsched(&himax->thread_attr, PTHREAD_EXPLICIT_SCHED);
    /* create the thread in a detached state. */
    pthread_attr_setdetachstate(&himax->thread_attr, PTHREAD_CREATE_DETACHED);

    /* describe the signal event, send a pulse
     * sigev_coid: the connection ID. this should be attached to the channel with
     *             which the pulse will be received.
     * sigev_code: a code to be interpreted by the pulse handler
     */
    himax->thread_event.sigev_notify = SIGEV_PULSE;
    himax->thread_event.sigev_coid = himax->thread_coid;
    himax->thread_event.sigev_code = HIMAX_PULSE_CODE;

    /* create interrupt handler thread */
    retval = pthread_create(&himax->isr_thread,
                        &himax->thread_attr,
                        (void *)himax_tp_recv_thread,
                        himax);
    if (EOK != retval) {
        mtouch_error(himax->log_name, "Fail to do pthread_create (error: %s)", strerror(errno));
        error_memory("Himax_Touch: Fail to do pthread_create (error: %s)", strerror(errno));
        return -EIO;
    }
    /* name a thread
     * if a thread is setting its own name, uses ThreadCtl()
     */
    pthread_setname_np(himax->isr_thread, "mtouch-himax-isr");

    // attach the given event to an interrupt source
    // assigning the interrupt vector number and the pointer of sigvent structure
    // that want to be delivered when this interrupt occurs
    //
    // before calling InterruptAttachEvent, it must request I/O privileges by calling ThreadCtl()
    if (EOK != himax_tp_intr_attach(himax)) {
       mtouch_error(himax->log_name, "Failed to attach to interrupt (error: %s)", strerror(errno));
       error_memory("Himax_Touch: Failed to attach to interrupt (error: %s)", strerror(errno));
       return -EIO;
    }

    MaskCnt = InterruptMask(himax->tp_intr, himax->tp_iid);
    mtouch_info(himax->log_name, "%s-%d>Interrupt handler registered MaskCnt[%d]", __FUNCTION__,__LINE__,MaskCnt);

    mtouch_info(himax->log_name, "Interrupt handler registered\n");

    return 0;
}

static void himax_mtouch_driver_remove(himax_dev_t *himax)
{
    if (!himax) {
        return;
    }

    pthread_cancel(himax->isr_thread);
    pthread_join(himax->isr_thread, NULL);

    pthread_cancel(himax->fidm_thread);
    pthread_join(himax->fidm_thread, NULL);

    if (himax->tp_iid >= 0) {
        InterruptDetach(himax->tp_iid);
        himax->tp_iid = -1;
    }

    if (himax->inputevents_hdl) {
        mtouch_driver_detach(himax->inputevents_hdl);
        himax->inputevents_hdl = NULL;
    }

    if (-1 != himax->i2c_fd) {
        i2c_close(himax->i2c_fd);
        himax->i2c_fd = -1;
    }

    if (himax->thread_chid >= 0) {
        ChannelDestroy(himax->thread_chid);
        himax->thread_chid = -1;
        ConnectDetach(himax->thread_coid);
        himax->thread_coid = -1;
    }

    if (himax->fidm_attach && himax->fidm_attach->chid >= 0) {
        ChannelDestroy(himax->fidm_attach->chid);
        himax->fidm_attach->chid = -1;
        ConnectDetach(himax->fidm_coid);
        himax->fidm_coid = -1;
        name_detach(himax->fidm_attach, 0);
        himax->fidm_attach = NULL;
    }

    if (himax->faceplate_coid) {
        name_close(himax->faceplate_coid);
        himax->faceplate_coid = -1;
    }

    free(himax);

}

himax_static int set_option(const char* option, const char* value, void* arg)
{
    himax_dev_t* himax = arg;
    if (0 == strcmp("width", option)) {
        return input_parse_unsigned(option, value, &himax->width);
    } else if (0 == strcmp("height", option)) {
        return input_parse_unsigned(option, value, &himax->height);
    } else if (0 == strcmp("i2c_dev_name", option)) {
        return input_parse_string(option, value, &himax->i2c_devname);
    } else if (0 == strcmp("i2c_speed", option)) {
        return input_parse_unsigned(option, value, &himax->i2c_speed);
    } else if (0 == strcmp("slave_addr", option)) {
        return input_parse_unsigned(option, value, &himax->i2c_slave_addr);
    } else if (0 == strcmp("intr_gpio_pin", option)) {
        return input_parse_unsigned(option, value, &himax->tp_intr_gpio);
    } else if (0 == strcmp("verbose", option)) {
        return input_parse_unsigned(option, value, &himax->verbose);
    } else if (0 == strcmp("num_touchpts", option)) {
        return input_parse_unsigned(option, value, &himax->max_touchpoints);
    } else if (0 == strcmp("pm_dev", option)) {
        return input_parse_string(option, value, &himax->pm_dev);
    } else if (0 == strcmp("win_height", option)) {
        return input_parse_unsigned(option, value, &himax->win_height);
    } else if (0 == strcmp("win_width", option)) {
        return input_parse_unsigned(option, value, &himax->win_width);
    } else if (0 == strcmp("disp_id", option)) {
        return input_parse_unsigned(option, value, &himax->disp_id);
    } else if (0 == strcmp("win_xpos", option)) {
        return input_parse_unsigned(option, value, &himax->win_xpos);
    } else if (0 == strcmp("win_ypos", option)) {
        return input_parse_unsigned(option, value, &himax->win_ypos);
    } else if (0 == strcmp("win_name", option)) {
        return input_parse_string(option, value, &himax->win_name);
    } else if (0 == strcmp("dev_ctrl_path", option)) {
        return input_parse_string(option, value, &himax->dev_ctrl_path);
    } else if (0 == strcmp("dev_status_path", option)) {
        return input_parse_string(option, value, &himax->dev_status_path);
    } else if (0 == strcmp("fidm_attach_point", option)) {
        return input_parse_string(option, value, &himax->fidm_attach_point);
    } else if (0 == strcmp("sync_point", option)) {
        return input_parse_unsigned(option, value, &himax->sync_point);
    } else {
        mtouch_error("Himax", "Invalid option: '%s'", option);
        return -1;
    }

    return -1;
}

void *mtouch_driver_init(const char* options)
{
    int ret = EOK;
    int retry_count = 0;
    int MaskCnt;
    int fd = -1;
    char str1[20] = {'\0'};
    himax_dev_t *himax = (himax_dev_t *) calloc(1, sizeof(himax_dev_t));   

    mtouch_info("Himax", "Himax driver init");

    if (InstallLocalErrMemReader(RINGBUFFER_SIZE) != 0)
    {
        mtouch_error("Himax", "errmem low priority worker thread creation failed !!!");
    }

    if(himax == NULL){
        mtouch_error("Himax", "touch_device Failed to allocate memory for device structure");
        error_memory("Himax_Touch: touch_device Failed to allocate memory for device structure");
        return NULL;
    }
    himax->HW_Init_status = false;
    himax->i2c_fd = -1;
    himax->i2c_devname = NULL;
    himax->i2c_speed = 400000;
    himax->i2c_slave_addr = 0;
    himax->width = H_PANEL_WIDTH;
    himax->height = H_PANEL_HEIGHT;
    himax->max_xfer = I2C_MAX_XFER;
    himax->inputevents_hdl = NULL;
    himax->tp_intr = 0xFFFF;
    himax->tp_intr_gpio = 0xFFFF;
    himax->tp_iid = -1;
    himax->isr_thread = 0;
    himax->thread_priority = 21;
    himax->win_height = 720;
    himax->win_width = 1280;
    himax->win_xpos = 0;
    himax->win_ypos = 0;
    himax->disp_id = 2;
    himax->verbose = 0;
    himax->pm_dev = PM_DEV_NODE;
    himax->max_touchpoints = HIMAX_MAX_NUM_TOUCHPTS;
    himax->thread_chid = -1;
    himax->thread_coid = -1;
    himax->dev_ctrl_path = "/dev/vcd/display-binder/control";
    himax->dev_status_path = "/dev/vcd/display-binder/status";
    himax->win_name = "qvm_window";
    himax->fidm_attach_point = NULL;
    himax->rtd_readflag = 0;
    himax->knob_rtd_readflag = 0;
    himax->knob_data = 0;
    himax->display_group_id = -1;
    himax->sync_point = 1;
    himax->hx_point_num = 0;
    himax->log_name = "Himax";

#ifdef BOSCH_RTC_2574268_HID_ENABLE_HIMAX_LOGGING_IMPROVEMENTS
    himax->fidm_RTD_rcvd_request_counter = 0;
    himax->fidm_RTD_prcsd_request_counter = 0;
    himax->fidm_RTD_rcvd_request_error_counter = 0;
    himax->fidm_RTD_prcsd_bkup_counter = 0;
    himax->Tp_pulse_received_counter = 0;
    himax->Tp_pulse_unmasked_counter = 0;
    himax->Tp_pulse_unmasked_rtd_counter = 0;
    himax->rtd_intr_flag = 0;
    himax->rtd_update_recvd_counter = 0;
    himax->rtd_update_prcsd_counter = 0;
#endif

    /* parses settings defined in graphics.conf */
    input_parseopts(options, set_option, himax);

    if((sprintf(str1, "himax_disp_id_%d", himax->disp_id)) < 0)
    {
       mtouch_error("Himax", "Failed to format display ID into log_string %s",strerror(errno));
       error_memory("Himax_Touch: %s: Failed to format display ID : %s", __FUNCTION__, strerror(errno));
    }
    else
       himax->log_name = strdup(str1);

    if (himax->max_touchpoints > HIMAX_MAX_NUM_TOUCHPTS) {
        mtouch_error("Himax", "num_touchpts exceeds limit of %d", HIMAX_MAX_NUM_TOUCHPTS);
        error_memory("Himax_Touch: num_touchpts exceeds limit of %d", HIMAX_MAX_NUM_TOUCHPTS);
        himax->max_touchpoints = HIMAX_MAX_NUM_TOUCHPTS;
    }


    /*do i2c bus setup*/
    if (NULL != himax->i2c_devname) {
        ret = himax_open_i2c_device(himax);
        if (ret < 0) {
            mtouch_error(himax->log_name,"failed to open i2c device: %d", ret);
            error_memory("Himax_Touch: failed to open i2c device: %d", ret);
            himax_mtouch_driver_remove(himax);
            return NULL;
        }
    } else {
        mtouch_error(himax->log_name,"i2c_devname is not provided");
        error_memory("Himax_Touch: i2c_devname is not provided");
        himax_mtouch_driver_remove(himax);
        return NULL;
    }

    /*hard reset*/
retry_himax_init:
    ret = himax_init(himax);
    if (ret != 0) {
       if (retry_count < MAX_HW_RETRY_ATTEMPTS) {
          retry_count++;
          mtouch_info(himax->log_name, "HW initialization failed, retrying %d\n",retry_count);
          if (usleep(500*1000) != 0) {
             mtouch_error(himax->log_name, "Failed to execute 500ms sleep %d\n", errno);
             error_memory("Himax_Touch: Failed to execute 500ms sleep %d\n", errno);
          }
          goto retry_himax_init;
       } else {
          mtouch_error(himax->log_name, "HW initialization failed, Moved to SUSPEND %d", ret);
          error_memory("Himax_Touch: HW initialization failed, Moved to SUSPEND %d", ret);
          himax->himax_pm_state = PM_STATE_SUSPEND;
          himax->HW_Init_status = false;
          //InterruptMask(himax->tp_intr, himax->tp_iid);
      }
    }
    else
    {
       /*Updated pm state to Resume on HW Init success, later driver is removed if there is any 
       *failure with threads creation,attachdriver.
       */
       himax->himax_pm_state = PM_STATE_RESUME;
       himax->HW_Init_status = true;
       mtouch_info(himax->log_name, "Himax_Touch: HW initialization Success");
    }


    /*create isr*/
    ret = himax_create_isr(himax);
    if (ret != 0) {
        mtouch_error(himax->log_name, "Fail to create ISR thread %d", ret);
        error_memory("Himax_Touch: Fail to create ISR thread %d", ret);
        himax_mtouch_driver_remove(himax);
        return NULL;
    }

    if(himax->HW_Init_status == true)
    {
        MaskCnt = InterruptUnmask(himax->tp_intr, himax->tp_iid);
        mtouch_info(himax->log_name,"%s-%d>UnMaskCnt[%d]", __FUNCTION__,__LINE__,MaskCnt);
    }

    if (himax->fidm_attach_point == NULL) {
        himax->fidm_attach_point = strdup (FIDM_TOUCH_ATTACH_POINT);
    }

#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
    /* Create channel for external messages */
    int fidm_chid = ChannelCreate(0); // Create a communication channel for external messages
    if (fidm_chid == -1) {
            mtouch_error(himax->log_name,"%s: ChannelCreate failed: %s", __FUNCTION__, strerror(errno));
            error_memory("Himax_Touch: %s: ChannelCreate failed: %s", __FUNCTION__, strerror(errno));
            return NULL; // Exit if channel creation fails
    }
    dispatch_t* fidm_dpp = dispatch_create_channel(fidm_chid, 0); // Create a dispatch structure for the external channel
    if (fidm_dpp == NULL) {
            mtouch_error(himax->log_name,"%s: dispatch_create_channel failed: %s", __FUNCTION__, strerror(errno));
            error_memory("Himax_Touch: %s: dispatch_create_channel failed: %s", __FUNCTION__, strerror(errno));
            ChannelDestroy(fidm_chid); // Destroy the channel if dispatch creation fails
            return NULL;
    }
    if ((himax->fidm_attach = name_attach(fidm_dpp, himax->fidm_attach_point, 0)) == NULL) {
            mtouch_error(himax->log_name,"%s: Message handler thread name_attach: %s", __FUNCTION__, strerror(errno));
            error_memory("Himax_Touch: %s: Message handler thread name_attach: %s", __FUNCTION__, strerror(errno));
            //dispatch_destroy(fidm_dpp); // Dispatch destroy commented out, ensure cleanup elsewhere
            ChannelDestroy(fidm_chid); // Destroy the channel if name attachment fails
            return NULL;
    }
#else
    /* Create channel for external messages */
    if ((himax->fidm_attach = name_attach(NULL, himax->fidm_attach_point, 0)) == NULL) {
        mtouch_error(himax->log_name, "%s: name_attach: %s", __FUNCTION__, strerror(errno));
        error_memory("Himax_Touch: %s: name_attach: %s", __FUNCTION__, strerror(errno));
        himax_mtouch_driver_remove(himax);
        return NULL;
    }
#endif
    himax->fidm_coid = ConnectAttach(0, 0, himax->fidm_attach->chid, _NTO_SIDE_CHANNEL, 0);
    if (-1 == himax->fidm_coid) {
        mtouch_error(himax->log_name, "%s: ConnectAttach: %s", __FUNCTION__, strerror(errno));
        error_memory("Himax_Touch: %s: ConnectAttach: %s", __FUNCTION__, strerror(errno));
        himax_mtouch_driver_remove(himax);
        return NULL;
    }

    if (EOK != pthread_create(&himax->fidm_thread, NULL, himax_ext_msg_handler, himax)) {
       mtouch_error(himax->log_name, "Failed to create the external message thread");
       error_memory("Himax_Touch: Failed to create the external message thread");
       himax_mtouch_driver_remove(himax);
       return NULL;
    }

    /*register to pm*/
    if (EOK != himax_mtouch_register_pm(himax)) {
        mtouch_error(himax->log_name, "registration with %s failed",himax->pm_dev );
        error_memory("Himax_Touch: registration with %s failed",himax->pm_dev );
    }
    else
        mtouch_info(himax->log_name, "registration with %s success",himax->pm_dev);

    if(attach_driver(himax) != EOK) {
        mtouch_error(himax->log_name, "Failed to attach to libinputevents");
        error_memory("Himax_Touch: Failed to attach to libinputevents");
        himax_mtouch_driver_remove(himax);
        return NULL;
    }

      /* create a window to forward events to guest OS */
    if (0 > himax_create_qvm_window(himax)) {
        mtouch_error(himax->log_name, "Failed to create window for qvm");
        error_memory("Himax_Touch: Failed to create window for qvm");
        himax_mtouch_driver_remove(himax);
        return NULL;
    }

    /* Display Group ID */
    ret = himax_get_display_grpid(himax);
    if(ret != 0)
    {
        mtouch_error(himax->log_name, "Failed to fetch Display Group Id: %d in init:retval: %d", himax->display_group_id, ret);
        error_memory("Himax_Touch: Failed to fetch Display Group Id: %d in init:retval: %d", himax->display_group_id, ret);
    }

    /*Connect to faceplate driver to send knob data*/
    ret = himax_connect_faceplate(himax);
    if (ret != 0) {
        mtouch_error(himax->log_name, "Failed to connect to faceplate driver, try again %d", ret);
        error_memory("Himax_Touch: Failed to connect to faceplate driver, try again %d", ret);
    }

    mtouch_info(himax->log_name, "himax touch driver initialized successfully");

    /* Create file to signal that the driver is initialized and ready to serve clients */
    if (himax->sync_point) {
        fd = open("/tmp/sync/mtouch_initialized", O_WRONLY | O_CREAT | O_TRUNC,
            S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
        if(fd == -1)
        {
            mtouch_error(himax->log_name, "%s failed to create sync file", __FUNCTION__);
            error_memory("Himax_Touch: %s failed to create sync file", __FUNCTION__);
        }
        else
        {
            close(fd);
        }
    } else {
        mtouch_info(himax->log_name, "diagnostics sync point not enabled %d", himax->sync_point);
    }

    return himax;

}

void mtouch_driver_fini(void* dev)
{
    himax_dev_t *himax = dev;

    himax_mtouch_driver_remove(himax);

}

